/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpRPCBackEnd.c,v 1.13.2.2 2001/06/21 23:52:06 sluu Exp $
____________________________________________________________________________*/
#include "pgpErrors.h"
#include "pgpRPCBackEndAPI.h"
#include "pgpRPCMsg.h"
#include "pgpMem.h"
#include "pgpThreads.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

static PGPBoolean sRPCEnabled = 0;
static PGPMutex_t sRPCMutex;

PGPBoolean
pgpRPCEnabled()
{
	return sRPCEnabled;
}

#if USE_SOLARIS_DOORS

/*
 * USE SOLARIS 2.6 "DOOR API" FOR IPC.
 */

#include <sys/door.h>

static int doorfd;

PGPError
pgpSDKInitBackEnd(PGPUInt32 backEndAPIVersion)
{
	PGPError err;

	(void)backEndAPIVersion;

	doorfd = open("/var/.pgpsdkdoor", O_RDWR);
	if (doorfd == -1) {
		return kPGPError_ImproperInitialization;
	}

	PGMutexCreate(&sRPCMutex, NULL);

	err = pgpConnect_backRPC();
	if (err != kPGPError_NoErr)
		return kPGPError_RPCFailed;

	sRPCEnabled = 1;
	return kPGPError_NoErr;
}

PGPInt32
pgpRPCSendPacket(PGPPackMsg *pkt, PGPPackMsg *rply_pkt, PGPBoolean okayToBlock)
{
	PGPInt32 reply_len, status, length;
	PGPError err;
	door_arg_t arg;
	char *reply_buf;
	int n;

	PGPMutexLock(&sRPCMutex);

	((PGPInt32 *)pkt->base)[2] = pkt->ptr;

	arg.data_ptr = (char *)pkt->base;
	arg.data_size = pkt->ptr;
	arg.desc_ptr = NULL;
	arg.desc_num = 0;
	arg.rbuf = NULL;
	arg.rsize = 0;
	n = door_call(doorfd, &arg);
	PGPFreeData(pkt->base);			/* Free Data-Send Buffer */
	if (n < 0) {
		PGPMutexUnlock(&sRPCMutex);
		return kPGPError_RPCFailed;
	}

	reply_buf = (char *)PGPNewData(PGPGetDefaultMemoryMgr(), arg.rsize, 0);
	reply_len = arg.rsize;
	memcpy(reply_buf, arg.rbuf, arg.rsize);

	munmap(arg.rbuf, arg.rsize);

	rply_pkt->base = reply_buf;
	rply_pkt->length = reply_len;
	rply_pkt->ptr = 0;
	status = unpack_int32(rply_pkt);
	length = unpack_int32(rply_pkt);
	if (IsPGPError(status)) {
		PGPFreeData(reply_buf);
	}
	PGPMutexUnlock(&sRPCMutex);
	return status;
}

void
pgpSDKCleanupBackEnd()
{
	close(doorfd);
	PGMutexDestroy(&sRPCMutex);
}

#endif

#if defined(sun)

/*
 * SOLARIS USES NAMED STREAMS
 */

#include <sys/stropts.h>

static int pipefd;

PGPError
pgpSDKInitBackEnd(PGPUInt32 backEndAPIVersion)
{
	PGPError err;
	int n, fd = 0;

	(void)backEndAPIVersion;

	pipefd = open("/var/.pgpsdkpipe", O_RDWR);
	if (pipefd == -1) {
		return kPGPError_ImproperInitialization;
	}

	err = pgpConnect_backRPC();
	if (err != kPGPError_NoErr)
		return kPGPError_RPCFailed;

	PGPMutexCreate(&sRPCMutex, NULL);

	sRPCEnabled = 1;
	return kPGPError_NoErr;
}

PGPInt32
pgpRPCSendPacket(PGPPackMsg *pkt, PGPPackMsg *rply_pkt, PGPBoolean okayToBlock)
{
	PGPInt32 reply_len, status, length;
	PGPError err;
	char *reply_buf;
	char header_buf[8];
	int n;

	PGPMutexLock(&sRPCMutex);

	((PGPInt32 *)pkt->base)[2] = pkt->ptr;

	n = write(pipefd, pkt->base, pkt->ptr);

	PGPFreeData(pkt->base);			/* Free Data-Send Buffer */
	if (n != pkt->ptr) {
		PGPMutexUnlock(&sRPCMutex);
		return kPGPError_RPCFailed;
	}
	/*
	 * Second 32-bit word in the reply is the length.  We need to read
	 * first the header and length, then allocate the rest of the packet,
	 * then we can finish reading from the service.
	 */
	reply_len = 0;
	do {
		reply_len += read(pipefd, header_buf + reply_len, sizeof(header_buf) - reply_len );
	} while( reply_len < sizeof(header_buf) );

	/* Read length from the buffer */
	length = *((PGPInt32 *)( header_buf + sizeof( PGPInt32 ) ) );

	reply_buf = (char *)PGPNewData(PGPGetDefaultMemoryMgr(), length, 0);
	if( reply_buf == NULL )
	{
		PGPMutexUnlock(&sRPCMutex);
		return kPGPError_OutOfMemory;
	}

	/* Make reply_buf be a flat buffer of all the data */
	pgpCopyMemory( header_buf, reply_buf, sizeof( header_buf ) );

	/* Now read the rest of the packet */
	reply_len = 0;
	do
	{
		reply_len += read(pipefd, 
			reply_buf + sizeof( header_buf ) + reply_len,
			length - sizeof( header_buf ) - reply_len );
	}
	while( reply_len < (PGPUInt32) length - sizeof( header_buf ) );

	rply_pkt->base = reply_buf;
	rply_pkt->length = reply_len;
	rply_pkt->ptr = 0;
	status = unpack_int32(rply_pkt);
	length = unpack_int32(rply_pkt);
	if (IsPGPError(status)) {
		PGPFreeData(reply_buf);
	}
	PGPMutexUnlock(&sRPCMutex);
	return status;
}

void
pgpSDKCleanupBackEnd()
{
	close(pipefd);
	PGPMutexDestroy(&sRPCMutex);
}

#endif


#if PGP_UNIX_LINUX

/*
 * LINUX USES UNIX-DOMAIN SOCKETS.
 */
static int sd;
#define PGPSOCKPATH "/var/.pgpsdksock"

PGPError
pgpSDKInitBackEnd(PGPUInt32 backEndAPIVersion)
{
	int n;
	struct sockaddr_un addr;
	PGPError err;

	if (backEndAPIVersion != kPGPsdkBackEndAPIVersion)
		return kPGPError_ImproperInitialization;

	sd = socket(PF_UNIX, SOCK_STREAM, 0);
	if (sd == -1)
		return kPGPError_ImproperInitialization;

	PGPMutexCreate(&sRPCMutex, NULL);

	bzero(&addr, sizeof(addr));
	addr.sun_family = AF_UNIX;
	strcpy(addr.sun_path, PGPSOCKPATH);

	n = connect(sd, (struct sockaddr *)&addr, sizeof(addr));
	if (n == -1)
		return kPGPError_ImproperInitialization;

	err = pgpConnect_backRPC();
	if (err != kPGPError_NoErr)
		return kPGPError_RPCFailed;

	sRPCEnabled = 1;
	return kPGPError_NoErr;
}

PGPInt32
pgpRPCSendPacket(PGPPackMsg *pkt, PGPPackMsg *rply_pkt, PGPBoolean okayToBlock)
{
	char *reply_buf;
	char header_buf[8];
	PGPUInt32 reply_len;
	PGPInt32 status, length;
	int n;

	PGPMutexLock(&sRPCMutex);

	((PGPInt32 *)pkt->base)[2] = pkt->ptr;

	n = write(sd, pkt->base, pkt->ptr);
	if (n != pkt->ptr) {
		printf("pgpRPCSendPacket(): write error\n");
		PGPMutexUnlock(&sRPCMutex);
		exit(1);
	}

	/*
	 * Second 32-bit word in the reply is the length.  We need to read
	 * first the header and length, then allocate the rest of the packet,
	 * then we can finish reading from the service.
	 */
	reply_len = 0;
	do {
		reply_len += read(sd, header_buf + reply_len, sizeof(header_buf) - reply_len );
	} while( reply_len < sizeof(header_buf) );

	/* Read length from the buffer */
	length = *((PGPInt32 *)( header_buf + sizeof( PGPInt32 ) ) );

	reply_buf = (char *)PGPNewData(PGPGetDefaultMemoryMgr(), length, 0);
	if( reply_buf == NULL )
	{
		PGPMutexUnlock(&sRPCMutex);
		return kPGPError_OutOfMemory;
	}

	/* Make reply_buf be a flat buffer of all the data */
	pgpCopyMemory( header_buf, reply_buf, sizeof( header_buf ) );

	/* Now read the rest of the packet */
	reply_len = 0;
	do
	{
		reply_len += read(sd, 
			reply_buf + sizeof( header_buf ) + reply_len,
			length - sizeof( header_buf ) - reply_len );
	}
	while( reply_len < (PGPUInt32) length - sizeof( header_buf ) );

	PGPFreeData(pkt->base);			/* Free Data-Send Buffer */
	rply_pkt->base = reply_buf;
	rply_pkt->length = reply_len;
	rply_pkt->ptr = 0;
	status = unpack_int32(rply_pkt);
	length = unpack_int32(rply_pkt);
	if (IsPGPError(status))
		PGPFreeData(reply_buf);
	PGPMutexUnlock(&sRPCMutex);
	return status;
}

void
pgpSDKCleanupBackEnd()
{
	close(sd);
	PGPMutexDestroy(&sRPCMutex);
}

#endif

#if !defined(linux) && !defined(sun) && !defined(WIN32)
/*
 *	Need to fix this stuff for AIX and HP-UX
 */
PGPError
pgpSDKInitBackEnd(PGPUInt32 backEndAPIVersion)
{
	return kPGPError_NoErr;
}

PGPInt32
pgpRPCSendPacket(PGPPackMsg *pkt, PGPPackMsg *rply_pkt, PGPBoolean okayToBlock)
{
	return 0;
}

void
pgpSDKCleanupBackEnd()
{
}
#endif
